<?php
// --- CONFIGURAÇÃO DO BANCO DE DADOS ---
$db_host = "localhost";
$db_port = 3306;
$db_user = "sgtm";
$db_pass = "sgtm";
$db_name = "estoque_db";

// --- CABEÇALHOS E MANIPULAÇÃO DE ERROS ---
header('Content-Type: application/json');
ini_set('display_errors', 1); // Desativar em produção
error_reporting(E_ALL);

// --- FUNÇÃO DE CONEXÃO ---
function conectar_db() {
    global $db_host, $db_port, $db_user, $db_pass, $db_name;
    try {
        $conn = new mysqli($db_host, $db_user, $db_pass, $db_name, $db_port);
        if ($conn->connect_error) {
            throw new Exception("Falha na conexão: " . $conn->connect_error);
        }
        $conn->set_charset("utf8mb4");
        return $conn;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// --- ROTEADOR DE AÇÕES ---
$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    switch($action) {
        case 'carregarDadosIniciais': carregarDadosIniciais(); break;
        case 'carregarHistoricoEstoque': carregarHistoricoEstoque(); break;
        case 'carregarDadosManutencao': carregarDadosManutencao(); break;
        case 'carregarHistoricoEquipamento': carregarHistoricoEquipamento($_GET['id']); break;
        case 'visualizarEstoqueCategoria': visualizarEstoqueCategoria($_GET['categoria']); break;
        default: echo json_encode(['success' => false, 'error' => 'Ação GET inválida']);
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch($action) {
        case 'adicionarCategoria': adicionarCategoria($_POST['nome']); break;
        case 'adicionarProduto': adicionarProduto($_POST); break;
        case 'adicionarEquipamento': adicionarEquipamento($_POST); break;
        case 'registrarMovimentacao': registrarMovimentacao($_POST); break;
        case 'marcarComoDefeito': marcarComoDefeito($_POST['id']); break;
        case 'salvarManutencao': salvarManutencao($_POST); break;
        default: echo json_encode(['success' => false, 'error' => 'Ação POST inválida']);
    }
}

// --- FUNÇÕES DE LÓGICA (GET) ---

function carregarDadosIniciais() {
    $conn = conectar_db();
    $response = [];
    
    // Carregar Categorias
    $result = $conn->query("SELECT nome FROM categorias ORDER BY nome");
    $categorias = [];
    while($row = $result->fetch_assoc()) { $categorias[] = $row['nome']; }
    $response['categorias'] = $categorias;

    // Carregar Produtos (para combobox de manutenção)
    $result = $conn->query("SELECT id, descricao, marca FROM produtos ORDER BY descricao");
    $produtos = [];
    while($row = $result->fetch_assoc()) { 
        $produtos[] = $row['id'] . ' - ' . $row['descricao'] . ($row['marca'] ? ' (' . $row['marca'] . ')' : '');
    }
    $response['listaProdutos'] = $produtos;

    $conn->close();
    echo json_encode(['success' => true, 'data' => $response]);
}

function carregarHistoricoEstoque() {
    $conn = conectar_db();
    $query = "SELECT m.id, p.descricao, m.tipo_movimentacao, m.quantidade, m.responsavel,
              m.destino_origem, DATE_FORMAT(m.data_hora, '%d/%m/%Y %H:%i') as data_hora
              FROM movimentacoes m JOIN produtos p ON m.produto_id = p.id 
              ORDER BY m.data_hora DESC LIMIT 200";
    $result = $conn->query($query);
    $historico = [];
    while($row = $result->fetch_assoc()) { $historico[] = $row; }
    $conn->close();
    echo json_encode(['success' => true, 'data' => $historico]);
}

function carregarDadosManutencao() {
    $conn = conectar_db();
    $response = [];

    // Equipamentos
    $query_eq = "SELECT id, tipo, marca, modelo, identificacao, status FROM equipamentos ORDER BY tipo, marca, modelo";
    $result_eq = $conn->query($query_eq);
    $equipamentos = [];
    while($row = $result_eq->fetch_assoc()) { $equipamentos[] = $row; }
    $response['equipamentos'] = $equipamentos;

    // Itens pendentes
    $pendentes = [];
    $query_pend_eq = "SELECT id, 'Equipamento' as tipo_item, identificacao, 'N/A' as responsavel, 'equipamento' as tag FROM equipamentos WHERE status = 'Com Defeito'";
    $result_pend_eq = $conn->query($query_pend_eq);
    while($row = $result_pend_eq->fetch_assoc()) { $pendentes[] = $row; }

    $query_pend_defeito = "SELECT d.id, p.descricao as tipo_item, d.numero_serie as identificacao, d.responsavel, 'defeito_estoque' as tag FROM defeitos d JOIN produtos p ON d.produto_id = p.id WHERE d.status = 'Pendente'";
    $result_pend_defeito = $conn->query($query_pend_defeito);
    while($row = $result_pend_defeito->fetch_assoc()) { $pendentes[] = $row; }
    
    $response['pendentes'] = $pendentes;

    $conn->close();
    echo json_encode(['success' => true, 'data' => $response]);
}

function carregarHistoricoEquipamento($id) {
    $conn = conectar_db();
    $query = $conn->prepare("
        SELECT 
            DATE_FORMAT(m.data_manutencao, '%d/%m/%Y %H:%i') as data, 
            m.responsavel, m.motivo, m.status_final,
            (SELECT GROUP_CONCAT(CONCAT(p.descricao, ' (', cu.quantidade_usada, ')') SEPARATOR '; ') 
             FROM componentes_usados cu JOIN produtos p ON cu.produto_id = p.id 
             WHERE cu.manutencao_id = m.id) as componentes
        FROM manutencoes m 
        WHERE m.equipamento_id = ? 
        ORDER BY m.data_manutencao DESC
    ");
    $query->bind_param("i", $id);
    $query->execute();
    $result = $query->get_result();
    $historico = [];
    while($row = $result->fetch_assoc()){ $historico[] = $row; }
    $query->close();
    $conn->close();
    echo json_encode(['success' => true, 'data' => $historico]);
}

function visualizarEstoqueCategoria($categoria_nome) {
    $conn = conectar_db();
    $query = $conn->prepare("SELECT p.id, p.descricao, p.unidade, p.quantidade, p.marca, p.lote FROM produtos p JOIN categorias c ON p.categoria_id = c.id WHERE c.nome = ? ORDER BY p.descricao");
    $query->bind_param("s", $categoria_nome);
    $query->execute();
    $result = $query->get_result();
    $produtos = [];
    while($row = $result->fetch_assoc()){ $produtos[] = $row; }
    $query->close();
    $conn->close();
    echo json_encode(['success' => true, 'data' => $produtos]);
}

// --- FUNÇÕES DE LÓGICA (POST) ---

function adicionarCategoria($nome) {
    $conn = conectar_db();
    $stmt = $conn->prepare("INSERT INTO categorias (nome) VALUES (?)");
    $stmt->bind_param("s", $nome);
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => "Categoria '$nome' adicionada."]);
    } else {
        echo json_encode(['success' => false, 'error' => $conn->error]);
    }
    $stmt->close();
    $conn->close();
}

function adicionarProduto($data) {
    $conn = conectar_db();
    // Obter ID da categoria
    $stmt_cat = $conn->prepare("SELECT id FROM categorias WHERE nome = ?");
    $stmt_cat->bind_param("s", $data['categoria_nome']);
    $stmt_cat->execute();
    $result_cat = $stmt_cat->get_result();
    if ($result_cat->num_rows == 0) {
        echo json_encode(['success' => false, 'error' => 'Categoria não encontrada.']);
        return;
    }
    $categoria_id = $result_cat->fetch_assoc()['id'];
    $stmt_cat->close();

    // Inserir produto
    $stmt = $conn->prepare("INSERT INTO produtos (categoria_id, descricao, unidade, marca, lote, quantidade) VALUES (?, ?, ?, ?, ?, 0)");
    $stmt->bind_param("issss", $categoria_id, $data['descricao'], $data['unidade'], $data['marca'], $data['lote']);
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'Produto adicionado com sucesso.']);
    } else {
        echo json_encode(['success' => false, 'error' => $conn->error]);
    }
    $stmt->close();
    $conn->close();
}

function adicionarEquipamento($data) {
    $conn = conectar_db();
    $stmt = $conn->prepare("INSERT INTO equipamentos (tipo, marca, modelo, identificacao) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("ssss", $data['tipo'], $data['marca'], $data['modelo'], $data['identificacao']);
     if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'Equipamento cadastrado com sucesso.']);
    } else {
        if ($conn->errno == 1062) {
             echo json_encode(['success' => false, 'error' => "A identificação '{$data['identificacao']}' já existe."]);
        } else {
             echo json_encode(['success' => false, 'error' => $conn->error]);
        }
    }
    $stmt->close();
    $conn->close();
}

function registrarMovimentacao($data) {
    $conn = conectar_db();
    $conn->begin_transaction();
    try {
        $produto_id = intval(explode(' - ', $data['produto'])[0]);
        $quantidade = floatval(str_replace(',', '.', $data['quantidade']));
        $tipo_mov = $data['tipo_movimentacao'];

        // Trava a linha do produto para evitar condição de corrida
        $stmt = $conn->prepare("SELECT quantidade FROM produtos WHERE id = ? FOR UPDATE");
        $stmt->bind_param("i", $produto_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $produto = $result->fetch_assoc();
        $quantidade_atual = $produto['quantidade'];
        $stmt->close();
        
        if ($tipo_mov == 'Saída') {
            if ($quantidade > $quantidade_atual) {
                throw new Exception("Estoque insuficiente.");
            }
            $nova_quantidade = $quantidade_atual - $quantidade;
        } else {
            $nova_quantidade = $quantidade_atual + $quantidade;
        }
        
        // Atualiza estoque
        $stmt_update = $conn->prepare("UPDATE produtos SET quantidade = ? WHERE id = ?");
        $stmt_update->bind_param("di", $nova_quantidade, $produto_id);
        $stmt_update->execute();
        $stmt_update->close();

        // Insere movimentação
        $stmt_mov = $conn->prepare("INSERT INTO movimentacoes (produto_id, tipo_movimentacao, quantidade, data_hora, responsavel, destino_origem) VALUES (?, ?, ?, NOW(), ?, ?)");
        $stmt_mov->bind_param("isdss", $produto_id, $tipo_mov, $quantidade, $data['responsavel'], $data['projeto']);
        $stmt_mov->execute();
        $stmt_mov->close();

        // Se for defeito
        if (isset($data['is_defeito']) && $data['is_defeito'] == 'true') {
            $numeros_serie = json_decode($data['numeros_serie']);
            $stmt_defeito = $conn->prepare("INSERT INTO defeitos (produto_id, data_registro, numero_serie, responsavel, status) VALUES (?, NOW(), ?, ?, 'Pendente')");
            foreach ($numeros_serie as $ns) {
                $stmt_defeito->bind_param("iss", $produto_id, $ns, $data['responsavel']);
                $stmt_defeito->execute();
            }
            $stmt_defeito->close();
        }

        $conn->commit();
        echo json_encode(['success' => true, 'message' => 'Movimentação registrada com sucesso.']);

    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    $conn->close();
}

function marcarComoDefeito($id) {
    $conn = conectar_db();
    $stmt = $conn->prepare("UPDATE equipamentos SET status = 'Com Defeito' WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'Equipamento marcado como defeituoso.']);
    } else {
        echo json_encode(['success' => false, 'error' => $conn->error]);
    }
    $stmt->close();
    $conn->close();
}

function salvarManutencao($data) {
    $conn = conectar_db();
    $conn->begin_transaction();

    try {
        $componentes = isset($data['componentes']) ? json_decode($data['componentes'], true) : [];
        
        // Validar e abater estoque dos componentes
        foreach ($componentes as $comp) {
            $produto_id = intval($comp['id']);
            $quantidade_usada = floatval($comp['quantidade']);

            $stmt_estoque = $conn->prepare("SELECT quantidade FROM produtos WHERE id = ? FOR UPDATE");
            $stmt_estoque->bind_param("i", $produto_id);
            $stmt_estoque->execute();
            $estoque_atual = $stmt_estoque->get_result()->fetch_assoc()['quantidade'];
            $stmt_estoque->close();

            if ($quantidade_usada > $estoque_atual) {
                throw new Exception("Estoque insuficiente para: " . $comp['nome']);
            }
            $nova_quantidade = $estoque_atual - $quantidade_usada;
            
            $stmt_update = $conn->prepare("UPDATE produtos SET quantidade = ? WHERE id = ?");
            $stmt_update->bind_param("di", $nova_quantidade, $produto_id);
            $stmt_update->execute();
            $stmt_update->close();

            // Registrar saída no histórico
            $stmt_mov = $conn->prepare("INSERT INTO movimentacoes (produto_id, tipo_movimentacao, quantidade, data_hora, responsavel, destino_origem) VALUES (?, 'Saída', ?, NOW(), ?, ?)");
            $destino = "Manutenção: " . $data['identificador'];
            $stmt_mov->bind_param("idss", $produto_id, $quantidade_usada, $data['responsavel'], $destino);
            $stmt_mov->execute();
            $stmt_mov->close();
        }

        // Inserir registro de manutenção
        $equipamento_id = ($data['type'] == 'equipamento') ? intval($data['item_id']) : null;
        $defeito_id = ($data['type'] == 'defeito_estoque') ? intval($data['item_id']) : null;
        
        $stmt_manut = $conn->prepare("INSERT INTO manutencoes (equipamento_id, defeito_id, data_manutencao, motivo, status_final, responsavel) VALUES (?, ?, NOW(), ?, ?, ?)");
        $stmt_manut->bind_param("iisss", $equipamento_id, $defeito_id, $data['motivo'], $data['status_final'], $data['responsavel']);
        $stmt_manut->execute();
        $manutencao_id = $conn->insert_id;
        $stmt_manut->close();
        
        // Associar componentes à manutenção
        $stmt_comp = $conn->prepare("INSERT INTO componentes_usados (manutencao_id, produto_id, quantidade_usada) VALUES (?, ?, ?)");
        foreach ($componentes as $comp) {
            $produto_id = intval($comp['id']);
            $quantidade_usada = floatval($comp['quantidade']);
            $stmt_comp->bind_param("iid", $manutencao_id, $produto_id, $quantidade_usada);
            $stmt_comp->execute();
        }
        $stmt_comp->close();

        // Atualizar status do item se resolvido
        if ($data['status_final'] == 'Resolvido') {
            if ($data['type'] == 'equipamento') {
                $stmt_status = $conn->prepare("UPDATE equipamentos SET status = 'OK' WHERE id = ?");
                $stmt_status->bind_param("i", $equipamento_id);
                $stmt_status->execute();
                $stmt_status->close();
            } else { // defeito_estoque
                $stmt_status = $conn->prepare("UPDATE defeitos SET status = 'Resolvido' WHERE id = ?");
                $stmt_status->bind_param("i", $defeito_id);
                $stmt_status->execute();
                $stmt_status->close();
            }
        }

        $conn->commit();
        echo json_encode(['success' => true, 'message' => 'Manutenção registrada com sucesso.']);

    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    $conn->close();
}
?>